// PCTPClient.cpp : Defines the entry point for the console application.
//
#include "stdafx.h"
#include "PCAN-ISO-TP.h"

#define USE_CAN_FD false
#define USE_EVENT false
#define USE_GETCH false

#define N_SA		((BYTE)0xF1)
#define N_TA_PHYS	((BYTE)0x13)
#define N_TA_FUNC	((BYTE)0x26)
#define N_RA		((BYTE)0x52)

#define TEST_PHYS_TRANSMISSION	1
#define TEST_FUNC_TRANSMISSION	1
#define TEST_BURST_TRANSMISSION 1

// A global value to display timing information
DWORD g_timeReference;

// Function called to clean opened ISO-TP channels
void consoleExit(void) {
	CANTP_Uninitialize(PCANTP_NONEBUS);
}
// Callback to handle closure of the console window
BOOL WINAPI HandlerRoutine(_In_ DWORD dwCtrlType) {
	consoleExit();
	return FALSE;
}

/// <summary>Outputs a summary of a received message.</summary>
/// <param name="Message">The Message.</param>
/// <param name="Status">Status of the Read function.</param>
/// <returns>0 if no error, otherwise 1</returns>
int DisplayMessage(TPCANTPMsg Message, TPCANTPStatus Status)
{
	int nbErr = 0; 

	if (Status == PCANTP_ERROR_OK) {
		printf("[%lu] ", GetTickCount() - g_timeReference);
		switch (Message.MSGTYPE)
		{
		case PCANTP_MESSAGE_REQUEST_CONFIRMATION:
			printf("Received confirmation from 0x%02x (to 0x%02x, with RA 0x%02x) - result: %i - %s\n",
				(int)Message.SA,
				(int)Message.TA,
				(int)Message.RA,
				(int)Message.RESULT,
				Message.RESULT != PCANTP_N_OK ? "ERROR !!!" : "OK !");
			break;
		case PCANTP_MESSAGE_INDICATION:
		case PCANTP_MESSAGE_INDICATION_TX:
			printf("  ... message pending from 0x%02x (to 0x%02x, with RA 0x%02x) LEN=%i - result: %i...\n",
				(int)Message.SA,
				(int)Message.TA,
				(int)Message.RA,
				(int)Message.LEN,
				(int)Message.RESULT);
			break;
		case PCANTP_MESSAGE_DIAGNOSTIC:
		case PCANTP_MESSAGE_REMOTE_DIAGNOSTIC:
		default:
			printf("Received message from 0x%02x (to 0x%02x, with RA 0x%02x) - result: %i - %s, Length: %i\n",
				(int)Message.SA,
				(int)Message.TA,
				(int)Message.RA,
				(int)Message.RESULT,
				Message.RESULT != PCANTP_N_OK ? "ERROR !!!" : "OK !",
				(int)Message.LEN);
			break;
		}
		if (Message.RESULT != PCANTP_N_OK)
			nbErr++;
	}
	else if (Status != PCANTP_ERROR_NO_MESSAGE) {
		nbErr++;
		printf("Got error on reading message %i\n", (int)Status);
	}
	
	return nbErr;
}

/// <summary>Reads a message.</summary>
/// <param name="Channel">The channel.</param>
/// <param name="nbMsgRead">Buffer to store the number of message actually read.</param>
/// <returns>number of errors</returns>
int ReadMessage(TPCANTPHandle Channel, int &nbMsgRead)
{
	TPCANTPMsg Message;
	TPCANTPStatus Status;
	TPCANTPTimestamp timestamp;

	int nbErr = 0;
	while((Status = CANTP_Read(Channel,&Message, &timestamp)) == PCANTP_ERROR_NO_MESSAGE)
		Sleep(10);
	nbErr = DisplayMessage(Message, Status);
	if (Status == PCANTP_ERROR_OK)
	{
		// if the message is not fully transmitted, read again
		if (Message.MSGTYPE == PCANTP_MESSAGE_INDICATION || Message.MSGTYPE == PCANTP_MESSAGE_INDICATION_TX)
			ReadMessage(Channel, nbMsgRead);	
		else
			nbMsgRead++;
	}
	return nbErr;
}



/// <summary>A function to transmit CAN-TP messages</summary>
/// <param name="Channel">Channel</param>
/// <param name="canIdType">canIdType</param>
/// <param name="msgType">CAN-TP Message Type (PCANTP_MESSAGE_DIAGNOSTIC)</param>
/// <param name="formatType">Format addressing type (see PCANTP_FORMAT_xxx)</param>
/// <returns>number of transmission errors</returns>
int transmit(TPCANTPHandle Channel, 
			  TPCANTPIdType canIdType,
			  TPCANTPMessageType msgType,
			  TPCANTPFormatType formatType,
			  bool doCanFd) 
{
	USHORT usIndex;
	TPCANTPStatus Status;
	TPCANTPMsg Message;
	BYTE lMaxSize;
	int nbErrPhys = 0;
	int nbErrFunc = 0;
	DWORD sleepTime = 0;
	int nbMsgRead = 0;

	// Output information message
	char lStr[50] = "??";
	switch (formatType)
	{
	case PCANTP_FORMAT_ENHANCED:
		strcpy_s(lStr, "Enhanced");
		break;
	case PCANTP_FORMAT_EXTENDED:
		strcpy_s(lStr, "Extended");
		break;
	case PCANTP_FORMAT_FIXED_NORMAL:
		strcpy_s(lStr, "Fixed Normal");
		break;
	case PCANTP_FORMAT_MIXED:
		strcpy_s(lStr, "Mixed");
		break;
	case PCANTP_FORMAT_NORMAL:
		strcpy_s(lStr, "Normal");
		break;
	}
	printf("\n\nTransmission of %s, %s message, %s addressing format", 
		PCANTP_ID_CAN_IS_EXTENDED(canIdType) ? "29 BIT CAN ID" : "11 BIT CAN ID",
		msgType == PCANTP_MESSAGE_DIAGNOSTIC ? "Diagnostic" : "Remote diagnostic", 
		lStr);
	if (USE_GETCH) {
		printf("\nPress <Enter> to continue...");
		getchar();
	}

	// Configuring of the Test message
	memset(&Message, 0, sizeof(TPCANTPMsg));
	Message.SA = N_SA;
	if (msgType == PCANTP_MESSAGE_REMOTE_DIAGNOSTIC || 
		formatType == PCANTP_FORMAT_ENHANCED)
		Message.RA = N_RA;
	Message.IDTYPE = canIdType;	
	Message.MSGTYPE = msgType;
	if (doCanFd) {
		Message.IDTYPE |= PCANTP_ID_CAN_FD;
		Message.IDTYPE |= PCANTP_ID_CAN_BRS;
	}
	Message.FORMAT = formatType;
	for (usIndex = 0 ; usIndex < 4095 ; usIndex++)
		Message.DATA[usIndex] = (BYTE)usIndex+1;

#if TEST_PHYS_TRANSMISSION
	// Transmitting data using Physical addressing
	printf("\nTransmitting data using Physical addressing\n\n");
	g_timeReference = GetTickCount();
	Message.TA = N_TA_PHYS;
	Message.TA_TYPE = PCANTP_ADDRESSING_PHYSICAL;   
	for (usIndex = 1 ; usIndex <= PCANTP_MESSAGE_MAX_LENGTH; usIndex++)
	{
		// each loop transmits a message with a data length incremented by 1
		Message.LEN = usIndex; 
		Status = CANTP_Write(Channel,&Message);
		Sleep(sleepTime);
		printf("\nWrite Diagnostic Physical Message: Length %i Status:%i\n", (int)usIndex, (int)Status);
		if (Status == PCANTP_ERROR_OK)
		{
			nbMsgRead = 0;
			nbErrPhys += ReadMessage(Channel, nbMsgRead);
			if (nbMsgRead != 1) {
				printf("Received unexpected messages (%d instead of 1)", nbMsgRead);
			}
		}
		else
		{
			nbErrPhys++;
		}
		if (usIndex == 100)	{ // skip and jump to maximum data length
			usIndex = PCANTP_MESSAGE_MAX_LENGTH - 1;
		}		
	}
#endif

#if TEST_BURST_TRANSMISSION
		printf("\nTransmitting data using Physical addressing in 'BURST' mode\n\n");
		g_timeReference = GetTickCount();
		Message.TA = N_TA_PHYS;
		Message.TA_TYPE = PCANTP_ADDRESSING_PHYSICAL;   
		// Transmitting data using Physical addressing WITHOUT waiting for response confirmation
		USHORT nbMsgSent = 0;
		for (usIndex = 1 ; usIndex <= PCANTP_MESSAGE_MAX_LENGTH; usIndex++)
		{
			// each loop transmits a message with a data length incremented by 1
			Message.LEN = usIndex; 
			Status = CANTP_Write(Channel,&Message);
			Sleep(sleepTime);
			printf("\nWrite Diagnostic Physical Message: Length %i Status:%i\n", (int)usIndex, (int)Status);
			if (Status == PCANTP_ERROR_OK)
			{
				nbMsgSent++;
			}
			else
			{
				nbErrPhys++;
			}
			if (usIndex == 50)	{	// skip and try bigger LEN
				usIndex = 2000;
			}
			if (usIndex == 2005)	{ // skip
				break;
			}
		}
		printf("\nReading confirmation for each transmitted messages in 'BURST' mode\n\n");
		for (nbMsgRead = 0; nbMsgRead < nbMsgSent;)
		{
			nbErrPhys += ReadMessage(Channel, nbMsgRead);
		}
		CANTP_Reset(Channel);
#endif

#if TEST_FUNC_TRANSMISSION
	// Transmitting data using Functional addressing 
	printf("\nTransmitting data using Functional addressing\n\n");
	g_timeReference = GetTickCount();
	Message.TA = N_TA_FUNC;	
	Message.TA_TYPE = PCANTP_ADDRESSING_FUNCTIONAL;
	// Reminder: Functional addressing shall only be supported 
	// for Single Frame communication.
	// Thus maximum size depends on Format Addressing Type.
	switch (formatType) {
		case PCANTP_FORMAT_EXTENDED:
		case PCANTP_FORMAT_MIXED:
			lMaxSize = 6;
			break;
		default:
			lMaxSize = 7;
	}
	for (usIndex = 1 ; usIndex < lMaxSize + 1 ; usIndex++)
	{
		Message.LEN = usIndex; 
		Status = CANTP_Write(Channel,&Message);
		Sleep(10);
		printf("\nWrite Diagnostic Functional Message: Length %i Status:%i\n", (int)usIndex, (int)Status);
		if (Status == PCANTP_ERROR_OK)
		{
			nbMsgRead = 0;
			nbErrFunc += ReadMessage(Channel, nbMsgRead);
			if (nbMsgRead != 1) {
				printf("Received unexpected messages (%d instead of 1)", nbMsgRead);
			}
		}
		else
		{
			nbErrFunc++;
		}
	}
#endif
	
	if (nbErrPhys > 0 || nbErrFunc > 0) {
		if (nbErrPhys > 0) {
			printf("\nERROR : %d errors occured.\n\n", nbErrPhys);
		}		
		if (nbErrFunc > 0) {
			printf("\nERROR : %d errors occured.\n\n", nbErrPhys);
		}
		if (USE_GETCH) {
			printf("\nPress <Enter> to continue...");
			getchar();
		}
	} 
	else {		
		printf("\nTransmissions successful !\n\n");
	}

	return nbErrFunc + nbErrPhys;
}

/// <summary>	Main entry-point for this application. </summary>
/// <param name="argc">	The argc. </param>
/// <param name="argv">	[in,out] If non-null, the argv. </param>
/// <returns>Exit status</returns>
int main(int argc, char* argv[])
{	
	TPCANTPHandle Channel;
	TPCANTPStatus Status;
	TPCANTPBaudrate baudrate;
	char * bitrateFd;
	bool useCanFd = false;
	bool doCanFd = false;
	BYTE param;
	int nbErr = 0;
	char buff[50] = {};

	printf("\nUsage: PCTPClient.exe [TPCANTPHandle] [USE_CAN_FD:0|1] [BTR0BTR1|BITRATE]\n");
	printf(" * CAN: PCTPClient.exe 0x51 0 0x1C \n");
	printf(" * CAN FD: PCTPClient.exe 0x51 1 \"f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1\" \n");
	printf("\n---------------------------------------------\n\n");

	// clear gracefully ISO-TP if console window is killed
	SetConsoleCtrlHandler(HandlerRoutine, TRUE);
	atexit(consoleExit);

	// Show version information
	CANTP_GetValue(0, PCANTP_PARAM_API_VERSION, &buff, 50);
	printf("PCAN-ISO-TP API Version : %s\n", buff);
	
	// 1st argument is the PCAN-Channel to use (PCAN-USB Channel 1)
	if (argc > 1)
		Channel = (TPCANTPHandle)strtol(argv[1], NULL, 0);
	else
		Channel = PCANTP_USBBUS1;
	// 2nd argument is CAN FD
	if (argc > 2)
		useCanFd = strtol(argv[2], NULL, 0) == 1;
	else
		useCanFd = USE_CAN_FD;
	// 3rd argument is bitrate (either CAN or CAN FD)
	baudrate = PCANTP_BAUD_500K;
	bitrateFd = "f_clock=80000000, nom_brp=10, nom_tseg1=12, nom_tseg2=3, nom_sjw=1, data_brp=4, data_tseg1=7, data_tseg2=2, data_sjw=1";
	if (argc > 3) {
		if (useCanFd)
			bitrateFd = argv[3];
		else 
			baudrate = (TPCANTPBaudrate)strtoul(argv[3], NULL, 0);
	}

	// Initializing of the ISO-TP Communication session
	printf("Connecting to channel 0x%02x...\n", Channel);
	if (!useCanFd) {
		printf(" * btr0btr1 = 0x%02x...\n", baudrate);
		Status = CANTP_Initialize(Channel, baudrate, 0, 0, 0);
	}
	else {
		printf(" * bitrateFd = \"%s\"...\n", bitrateFd);
		Status = CANTP_InitializeFD(Channel, bitrateFd);
	}
	printf(" -> Initialize CANTP: %x\n", (int)Status);
	if (Status != PCANTP_ERROR_OK) {
		// initialization failed: abort
		printf(" -> Initialization failed, exiting...\n");
		CANTP_Uninitialize(Channel);
		printf("\n\nPress <Enter> to quit...");
		getchar();
		return -1;
	}

	// configure the test session to show notifications of pending multi-frame CAN messages (hidden by default)
	param = PCANTP_MSG_PENDING_SHOW;
	CANTP_SetValue(Channel,PCANTP_PARAM_MSG_PENDING, &param, 1);
	// configure the test session to use can data padding with a specific padding value
	param = PCANTP_CAN_DATA_PADDING_ON;
	CANTP_SetValue(Channel, PCANTP_PARAM_CAN_DATA_PADDING, &param, 1);
	param = 0x99;
	CANTP_SetValue(Channel, PCANTP_PARAM_PADDING_VALUE, &param, 1);
	// set the default TX_DL size (i.e. the default CAN-FD DLC)
	param = 0x0D;
	CANTP_SetValue(Channel, PCANTP_PARAM_CAN_TX_DL, &param, 1);
	// log CAN frames (0=disabled, 1=enabled)
	param = 0;
	Status = CANTP_SetValue(Channel, PCANTP_PARAM_DEBUG, &param, 1);
	
	for (int i = 0; i < 2; i++) {
		switch (i) {
		case 0:
			// first loop will test CAN-FD frames even if it was not initialized with Fd
			doCanFd = true;
			break;
		case 1:
			// second loop tests standard CAN frames
			doCanFd = false;
			break;
		}

		// Transmit with 29 Bit CAN identifiers with non-default J1939 priority
		nbErr += transmit(Channel, PCANTP_ID_CAN_GET_29B(0x00), PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_FIXED_NORMAL, doCanFd);
		nbErr += transmit(Channel, PCANTP_ID_CAN_GET_29B(0x07), PCANTP_MESSAGE_REMOTE_DIAGNOSTIC, PCANTP_FORMAT_MIXED, doCanFd);
		nbErr += transmit(Channel, PCANTP_ID_CAN_GET_29B(0x01), PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_ENHANCED, doCanFd);

		// Transmit with 29 Bit CAN identifiers (and no CAN ID mapping is required)
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_FIXED_NORMAL, doCanFd);	
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC, PCANTP_FORMAT_MIXED, doCanFd);	

		// Transmit with enhanced diagnostics 29 bit CAN identifiers (ISO-15765-3 8.3)
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_ENHANCED, doCanFd);

		// Transmit with 11 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Netword Addressing Information
		// for each transmission, 3 mappings are added :
		//	- 2 for physical addressing, one defining SA->TA and the other TA->SA
		//	- 1 for functional addressing defining SA->TA
		//

		// 11 bit can ID, normal format addressing (diagnostic message mandatory)
		CANTP_AddMapping(Channel, 0xA1, 0xA2,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA2, 0xA1,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, 0x00);
		nbErr += transmit(Channel, PCANTP_ID_CAN_11BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_NORMAL, doCanFd);
		// 11 bit can ID, extended format addressing, diagnostic message
		CANTP_AddMapping(Channel, 0xB1, 0xB2,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xB2, 0xB1,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xB5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, 0x00);
		nbErr += transmit(Channel, PCANTP_ID_CAN_11BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_EXTENDED, doCanFd);
#if 0	// WARNING: this 2004 mapping was not valid
		// 11 bit can ID, extended format addressing, remote diagnostic message
		CANTP_AddMapping(Channel, 0xC1, 0xC2,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xC2, 0xC1,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xC5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, N_RA);
		nbErr += transmit(Channel, PCANTP_ID_CAN_11BIT, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC, PCANTP_FORMAT_EXTENDED, doCanFd);
#endif
		// 11 bit can ID, mixed format addressing (remote diagnostic message mandatory)
		CANTP_AddMapping(Channel, 0xD1, 0xD2,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_MIXED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xD2, 0xD1,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_MIXED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xD5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_11BIT, PCANTP_FORMAT_MIXED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, N_RA);
		nbErr += transmit(Channel, PCANTP_ID_CAN_11BIT, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC, PCANTP_FORMAT_MIXED, doCanFd);

		CANTP_RemoveMapping(Channel, 0xA1);
		CANTP_RemoveMapping(Channel, 0xA2);
		CANTP_RemoveMapping(Channel, 0xB1);
		CANTP_RemoveMapping(Channel, 0xB2);
		CANTP_RemoveMapping(Channel, 0xC1);
		CANTP_RemoveMapping(Channel, 0xC2);
		CANTP_RemoveMapping(Channel, 0xD1);
		CANTP_RemoveMapping(Channel, 0xD2);

		// Transmit with 29 Bit CAN identifiers
		// those transmissions require a Mapping between CAN ID and Netword Addressing Information
		// for each transmission, 3 mappings are added 2 for physical addressing and 
		// the other for functional addressing.
		//

		// 29 bit can ID, normal format addressing (diagnostic message mandatory)
		CANTP_AddMapping(Channel, 0xA123A1, 0xA123A2,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA123A2, 0xA123A1,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA123A5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_NORMAL, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, 0x00);
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_NORMAL, doCanFd);
		// 29 bit can ID, extended format addressing, diagnostic message
		CANTP_AddMapping(Channel, 0xA123C1, 0xA123C2,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA123C2, 0xA123C1,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, 0x00);
		CANTP_AddMapping(Channel, 0xA123C5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, 0x00);
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_DIAGNOSTIC, PCANTP_FORMAT_EXTENDED, doCanFd);
#if 0	// WARNING: this 2004 mapping was not valid
		// 29 bit can ID, extended format addressing, remote diagnostic message
		CANTP_AddMapping(Channel, 0xA123D1, 0xA123D2,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_PHYS, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xA123D2, 0xA123D1,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_TA_PHYS, N_SA, PCANTP_ADDRESSING_PHYSICAL, N_RA);
		CANTP_AddMapping(Channel, 0xA123D5, CAN_ID_NO_MAPPING,
			PCANTP_ID_CAN_29BIT, PCANTP_FORMAT_EXTENDED, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
			N_SA, N_TA_FUNC, PCANTP_ADDRESSING_FUNCTIONAL, N_RA);
		nbErr += transmit(Channel, PCANTP_ID_CAN_29BIT, PCANTP_MESSAGE_REMOTE_DIAGNOSTIC, PCANTP_FORMAT_EXTENDED, doCanFd);
#endif
	}

	if (nbErr > 0) {
		printf("\nERROR : a total of %d errors occured.\n\n", nbErr);		
		printf("\n\nPress <Enter> to quit...");
		getchar();
	} 
	else {		
		printf("\nALL Transmissions succeeded !\n\n");
	}

	CANTP_Uninitialize(Channel);
	return 0;
}